/*
    Single-Use Camera Reader (c)2003 T. R. Gipson. 
    Program home page: http://cexx.org/dakota/

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

//---------------------------------------------------------------------------
#include <vcl.h>
#pragma hdrstop
#include "usb.h"
#include "Unit1.h"
#include <stdio.h>
#include <dos.h>       /* Used only for the 'sleep' function...if you decide to fix this with something more portable, this can be removed. */

//---------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"
TForm1 *Form1;
TForm1 *AboutForm;
TForm1 *MemoryWatcherForm;
TForm1 *MemEdit;
TForm1 *HowManyForm;
//---------------------------------------------------------------------------

/* This code may generate warnings depending on your compiler... as far as I know, they are all harmless. */
/* But if they bother you, feel free to fix 'em and submit a patch :-) */


/* Globals*/
struct usb_dev_handle* cam;     // Device handle to the camera
int configuration, interface, altsetting=-1;
int bulk_endpoint=-1;           // The endpoint (or 'pipe') used in all bulk transfers. The exact endpoint to use will be determined
                                // while walking through the device's properties.

/* This is from the FAT32 Directory Entry Structure, so it's a superset of FAT12 */
/* Most of this data is unused by this program, so we can ignore the endian-ness */
/* Of course, we could just as easily use a char[] buffer here, since all we want is the filename and size */
typedef struct {
    unsigned char Name[11];    /* not zero terminated */
    unsigned __int8 Attr;
    unsigned __int8 NTRes;         /* reserved */
    unsigned __int8 CrtTimeTenth;
    unsigned __int16 CrtTime;
    unsigned __int16 CrtDate;
    unsigned __int16 LstAccDate;
    unsigned __int16 FstClusHI;    /* Used only with FAT32 */
    unsigned __int16 WrtTime;
    unsigned __int16 WrtDate;
    unsigned __int16 FstClusLO;
    unsigned __int32 FileSize;
} toc_entry_type;


__fastcall TForm1::TForm1(TComponent* Owner)
        : TForm(Owner)
{
        StatusBar->SimpleText="Detecting USB Devices";
        OpenCam();

}
//---------------------------------------------------------------------------

/* Note about USB transfers: All control transfers are either in (device -> host) or out (host -> device) */
/* and in all cases, type=vendor and recipient=interface */
/* This data is passed in a bitfield which is the 2nd parameter of usb_control_msg.: */
/* 0xc1 = In  (type=vendor and recipient=interface) */
/* 0x41 = Out (type=vendor and recipient=interface) */

/* See http://www.maushammer.com/systems/dakotadigital/usb-commands.html for excellent documentation of the */
/* camera's USB commands. */

char get_ready_status()
{
        char buf[1];
        //In, get status (0x21)
        usb_control_msg(cam, 0xc1, 0x21, 0x0000, 0x0000, buf, 1, 5000);
        return buf[0];
}

int wait_for_ready()
{
        // FIXME: Possibly find better delay method than (almost certainly non-portable) MS-DOS sleep call
        int timeleft = 5;  /* 5 seconds */

        while(!get_ready_status() && --timeleft)
        {
                _sleep(1);  // Stalls for 1 second - really should find a better/faster way to wait!
        }
        if(!timeleft)
        {
                printf("timeout waiting for ready!!\n");
                fflush(stdout);
                return true;
        }
        return false;
}

toc_entry_type* get_toc(unsigned int n_toc_entries)
{
    toc_entry_type* buf;
    int err=0;
    unsigned int toc_size = n_toc_entries * sizeof(toc_entry_type);
    unsigned long endpoint_size;

    if (toc_size % 512 != 0)
    {
        toc_size = ((toc_size / 512) + 1) * 512;
    }
    buf = (toc_entry_type*) malloc(toc_size);
    if(buf == NULL)
    {
        return NULL;
    }
    memset(buf, 0, toc_size);
    endpoint_size = toc_size;


    /* If a bulk transfer is interrupted for any reason, there will still be the remaining data on the pipe */
    /* (regardless of what you requested), even if the camera is 'reset'. The only way to clear this seems to be */
    /* to unplug it and plug it back in */
    /* So, the command below will do something (as the libusb-win32 developer assures me :-), but clear the pipe isn't it. */

    err = usb_resetep(cam, bulk_endpoint);
    if (err)
    {
      printf("transferData: ResetPipe fail. ret = %08x\n", err);
    }

    // In, read number of TOC entries (0x54)
    usb_control_msg(cam, 0xc1, 0x54, n_toc_entries, 0x0001, NULL, 0, 5000);


    /* probably unnecessary: */
    err = usb_clear_halt(cam, bulk_endpoint);
    if (err)
    {
       printf("transferData: ClearPipeStall fail. ret = %08x\n", err);
       fflush(stdout);
    }

    wait_for_ready();

    /* Slurp all the TOC data in one bulk transfer */
    err = usb_bulk_read(cam, bulk_endpoint, (char*)buf, endpoint_size, 5000);
    printf("Reading TOC returned: %d\n", err);

    for(int i=0; i<toc_size; i++)
    {
        if(i%32 == 0) printf("\n%04x: ", i);
        printf ("%02x ", (unsigned int) ((char *)buf)[i]);
    }
    printf("\n");

    //getch();
    return buf;
}


void cam_reset()
{
        /* Out (host -> device), type=vendor, recipient=interface */
        /* Timeout of 5 seconds */
        usb_control_msg(cam, 0x41, 0x00, 0x0001, 0x2306, NULL, 0, 5000);
        usb_control_msg(cam, 0x41, 0x00, 0x0000, 0x0d04, NULL, 0, 5000);
        usb_control_msg(cam, 0x41, 0x1e, 0x0000, 0x0000, NULL, 0, 5000);

}

void cam_enable()
{
       char buf[9];
       char writebyte;
       unsigned long bcd_serial;
       unsigned long return_value;

        /* Get camera serial number - this is used in the camera-enable handshake (or, you can cheat--see cam_force_enable() )*/
        /*In, read serial (0x2d) */
        usb_control_msg(cam, 0xc1, 0x2d, 0x0000, 0x0001, buf, 8, 5000);

    buf[8] = '\0';      /* print ascii version */
    printf("Raw serial number %s\n", buf);



    bcd_serial = ((buf[0] & 0x0f) << 28) |
                 ((buf[1] & 0x0f) << 24) |
                 ((buf[2] & 0x0f) << 20) |
                 ((buf[3] & 0x0f) << 16) |
                 ((buf[4] & 0x0f) << 12) |
                 ((buf[5] & 0x0f) << 8) |
                 ((buf[6] & 0x0f) << 4) |
                 ((buf[7] & 0x0f) << 0);

    /* I would have to dig up my old Digits book to see if the above is actually a BCD conversion or not... */
    /* 'Til then, take it with a grain of salt (or, dig up your own digital systems reference...) */

    printf("BCD serial number %0x\n", (int) bcd_serial);

    return_value = ~bcd_serial << 2;
    printf("return value = %08x\n", (int) return_value);

    /* Note: Your compiler may give you warnings here, e.g. 'This conversion may lose significant digits'. */
    /* These can be ignored, since that's pretty much the idea. */


    writebyte = return_value & 0xff;
    /*Out*/
    usb_control_msg(cam, 0x41, 0x2d, 0x0000, 0x0000, &writebyte, 1, 5000);

    writebyte = (return_value >> 8) & 0xff;
    /*Out*/
    usb_control_msg(cam, 0x41, 0x2d, 0x0000, 0x0001, &writebyte, 1, 5000);


    writebyte = (return_value >> 16) & 0xff;
    /*Out*/
    usb_control_msg(cam, 0x41, 0x2d, 0x0000, 0x0002, &writebyte, 1, 5000);


    writebyte = (return_value >> 24) & 0xff;
    /*Out*/
    usb_control_msg(cam, 0x41, 0x2d, 0x0000, 0x0003, &writebyte, 1, 5000);

}

void cam_force_enable()
{
        /* Clear the comparison result at 0x0ee7, making the cam think the compare succeeded */
        /* This method is not really recommended, since the memory location of this byte could change in future firmware revisions...*/
        usb_control_msg(cam, 0x41, 0x00, 0x0000, 0x0ee7, NULL, 0, 5000);
}


unsigned int get_n_toc_entries()
{

    unsigned char buf[2];

    /* In, get n_toc_entries (0x54) */
    usb_control_msg(cam, 0xc1, 0x54, 0x0000, 0x0000, buf, 2, 5000);

    return (buf[1] << 8) | buf[0];
}

int TForm1::DownloadPictures()
{
        StatusBar->SimpleText="Reading Table Of Contents";
        StatusBar->Repaint();

        unsigned int numPictures=get_n_toc_entries();

        //printf("Camera contains %d pictures.\n", numPictures);
        toc_entry_type* toc=get_toc(numPictures);

        int result=SaveDialog1->Execute();
        Form1->Repaint();

        if (result==0)
        {
                // Save dialogue cancelled
                return 1;
        }

        for(int i=0; i<numPictures; i++)
        {
                /* This would all be in its own function, but then you can't update the status bar */

                toc_entry_type toc_entry=toc[i];

                char outfilename[13], extended_filename[256];
                FILE *fout;
                char * file_buf;
                char status[256]; /* character buffer for status bar display */
                int err;
                int i, j;
                unsigned int file_number;
                unsigned long exact_filesize, rounded_filesize;
                unsigned long endpoint_size;

                if (toc_entry.Name[0] == 0xe5) /* I don't think the camera returns these - Probably unnecessary */
                {
                        printf("Deleted file found\n");
                        return 2;
                }

                /* convert directory entry into a proper filename */
                for (i=0, j=0; i<8 && toc_entry.Name[i] != ' '; i++)
                {
                        outfilename[j++] = toc_entry.Name[i];
                }
                outfilename[j++] = '.';
                for (i=8; i<11 && toc_entry.Name[i] != ' '; i++)
                {
                        outfilename[j++] = toc_entry.Name[i];
                }
                outfilename[j++] = '\0';

                /* Sanity check - make sure filename ends with .JPG, not binary garbage */
                if (toc_entry.Name[8] != 'J' || toc_entry.Name[9] != 'P' || toc_entry.Name[10] != 'G')
                {
                        printf("TOC sanity check failed! Try unplugging and replugging the camera, then running this program again.\n");
                        Application->MessageBox("TOC sanity check failed! Try unplugging and replugging the camera, then running this program again.",NULL, MB_OK);
                        usb_release_interface(cam, interface);
                        usb_close(cam);
                        exit(2);
                }

                exact_filesize=toc_entry.FileSize;
                //exact_filesize = ((toc_entry.FileSize & 0xff000000) >> 24) |     /* proper endian to find filesize */
                //                 ((toc_entry.FileSize & 0x00ff0000) >> 8) |      /* Intel machines don't need this; just cast as a long int and forget about it */
                //                 ((toc_entry.FileSize & 0x0000ff00) << 8) |
                //                 ((toc_entry.FileSize & 0x000000ff) << 24);
                rounded_filesize = (exact_filesize / 0x4000) * 0x4000 + ((exact_filesize % 0x4000) ? 0x4000 : 0);

                //getch();
                printf("\n--------------------------------------------------------------------\n");
                printf("Filename %s, size=%u (0x%x)\n", outfilename, (int) exact_filesize, (int) exact_filesize);

                int kb_filesize=exact_filesize/1024.0;
                sprintf(status, "Downloading file %s, %d kb", outfilename, kb_filesize);
                StatusBar->SimpleText=status;
                StatusBar->Repaint();

                //printf("Downloading file %s, size %u bytes (%g KB)\n", outfilename, (int) exact_filesize,
                //    exact_filesize / 1024.0);

                fflush(stdout);

                file_buf = (char *) malloc(rounded_filesize);
                if(file_buf==NULL)
                {
                        printf("unable to allocate %u bytes for file\n", (int) rounded_filesize);
                        return 3;
                }
                memset(file_buf, 0, rounded_filesize);

                /* Extract number from filename. Example filename:  "DSC_0003JPG". */
                sscanf(&(toc_entry.Name[4]), "%d", &file_number);


                /* Out, put the data on the output endpoint */
                usb_control_msg(cam, 0x41, 0x54, file_number, 0x0002, NULL, 0, 5000);

                wait_for_ready();

                endpoint_size = rounded_filesize;
                /* Slurp in file data from the endpoint */
                err = usb_bulk_read(cam, bulk_endpoint, (char*)file_buf, endpoint_size, 5000);

                if (err < 0)
                {
                        printf("Unable to do bulk read (%08x)\n",err);
                }
                // printf("Read pipe err = %08x, size = %d of %d:  ", err, (int) endpoint_size, (int) rounded_filesize);

                sprintf(extended_filename, "%s%s", SaveDialog1->FileName.c_str(), outfilename);

                fout=fopen(extended_filename, "rb");
                if (fout)
                {
                        /* File exists! */
                        err=Application->MessageBox("File already exists. Overwrite it?", NULL, MB_YESNOCANCEL);
                }
                fclose(fout);

                if (err==2)     /* 'Cancel' button */
                {
                        return 5; /* (abort file downloading entirely) */
                }
                else if (err==7) /* 'No' button */
                {
                        /* Later I'll think of something to do here, for now we just skip that file */
                }
                else
                {

                        fout = fopen(extended_filename, "wb");
                        if(fout == NULL)
                        {
                                printf("unable to open output file '%s'\n", extended_filename);
                                Application->MessageBox("Unable to open output file for writing.",NULL, MB_OK);
                                free(file_buf);
                                return 4;
                        }
                        fwrite(file_buf, exact_filesize, 1, fout);
                        free(file_buf);
                        fclose(fout);

                }
        }
        return 0;
}


int TForm1::OpenCam()
{
    	struct usb_bus *busses;
        //toc_entry_type* toc;
        int error=0;

        usb_set_debug(3); /* Enable libusb debug messages - recommended since both of these are still fairly 'green', not-widely-tested software */
                        /* Use something like sysinternals.com's DebugView to view the debug messages */
    	usb_init();
    	usb_find_busses();
    	usb_find_devices();
    	busses = usb_get_busses();


        //if (busses==0)
        //{
        //        /* TESTME */
        //        //StatusBar->SimpleText=;
        //        Application->MessageBox("No USB bus detected - make sure drivers are installed correctly","Error", MB_OK);
        //}

    	struct usb_bus *bus;

    	for (bus = busses; bus; bus = bus->next) {
    		struct usb_device *dev;

    		for (dev = bus->devices; dev; dev = dev->next) {
    			if (
                        (dev->descriptor.idVendor == 0x0461 && dev->descriptor.idProduct == 0x0819) ||       // Walgreens (0461/0819)
                        (dev->descriptor.idVendor == 0x04fc && dev->descriptor.idProduct == 0xffff) ) {      // Ritz Camera Dakota (04FC/FFFF)
                                //printf("Found camera...\n");
                                char message[512];
                                sprintf(message, "Found camera with USB id %04X/%04X",dev->descriptor.idVendor, dev->descriptor.idProduct);
                                StatusBar->SimpleText=message;
                                cam=usb_open(dev);
                                if (!cam)
                                {
                                        //printf("Error opening device.\n");
                                        StatusBar->SimpleText="Error opening device.";
                                }

                                /* Walk through the possible configs, etc. */

                                //printf("This device has %d possible configuration(s).\n", dev->descriptor.bNumConfigurations);
                                for (int c = 0; c < dev->descriptor.bNumConfigurations; c++)
                                {
                                        //printf("Looking at configuration %d...This configuration has %d interfaces.\n", c, dev->config[c].bNumInterfaces);
                                        for(int i=0; i<dev->config[c].bNumInterfaces; i++)
                                        {
                                                //printf("  Looking at interface %d...This interface has %d altsettings.\n", i, dev->config[c].interface[i].num_altsetting);
                                                for (int a=0; a < dev->config[c].interface[i].num_altsetting; a++)
                                                {
                                                        //printf("    Looking at altsetting %d...This altsetting has %d endpoints.\n", a, dev->config[c].interface[i].altsetting[a].bNumEndpoints);
                                                        for (int e=0; e < dev->config[c].interface[i].altsetting[a].bNumEndpoints; e++)
                                                        {
                                                                //printf("      Endpoint %d: Address %02xh, attributes %02xh ", e, dev->config[c].interface[i].altsetting[a].endpoint[e].bEndpointAddress, dev->config[c].interface[i].altsetting[a].endpoint[e].bmAttributes);
                                                                char attribs = (dev->config[c].interface[i].altsetting[a].endpoint[e].bmAttributes & 3);
                                                                switch (attribs)
                                                                {
                                                                        case 0: //printf("(Control) Ignore this: ");
                                                                                break;
                                                                        case 1: //printf("(Isochronous) ");
                                                                                break;
                                                                        case 2: //printf("(Bulk) ");
                                                                                /* Found the correct configuration, interface etc... it has bulk endpoints! */
                                                                                configuration=c;
                                                                                interface=i;
                                                                                altsetting=a;
                                                                                break;
                                                                        case 3: //printf("(Interrupt) ");
                                                                                break;
                                                                        default: ;//printf("ERROR! Got an illegal value in endpoint bmAttributes\n");
                                                                }

                                                                char dir = (dev->config[c].interface[i].altsetting[a].endpoint[e].bEndpointAddress & 0x80);
                                                                switch (dir)
                                                                {
                                                                        case 0x00: //printf("(Out)\n");
                                                                                // Do nothing in this case
                                                                                break;

                                                                        case 0x80: //printf("(In)\n");
                                                                                   if ((dev->config[c].interface[i].altsetting[a].endpoint[e].bmAttributes & 0x03) == 2) /* Make sure it's a *bulk* endpoint */
                                                                                   {
                                                                                        // Found the correct endpoint to use for bulk transfer; use its ADDRESS
                                                                                        bulk_endpoint=dev->config[c].interface[i].altsetting[a].endpoint[e].bEndpointAddress;
                                                                                   }
                                                                                break;
                                                                        default: ;//printf("ERROR! Got an illegal value in endpoint bEndpointAddress\n");
                                                                }
                                                        }

                                                }
                                        }
                                }

                                /* Now what? Claim an interface(s), I guess */
                                if (bulk_endpoint<0)
                                {
                                        //printf("No valid interface found!\n");
                                        StatusBar->SimpleText="Found camera, but no valid interface!";
                                        return 1;
                                }


                                /* Set default configuration */
                                error=usb_set_configuration(cam, configuration+1);  // 1? This sounds wrong, but oh well...
                                //printf("Set config: %d\n", error);

                                error=usb_claim_interface(cam, interface);

                                //printf("Found bulk endpoint %d on Configuration %d Interface %d Altsetting %d\n", bulk_endpoint, configuration, interface, altsetting);

                                /* Set alt. setting to 1...or is it 0 in libusb? */
                                error=usb_set_altinterface(cam, altsetting);
                                //printf("Set alt. interface: %d\n", error);

                                cam_reset();
                                cam_enable();
                                NumPicturesLabel->Caption=get_n_toc_entries();
                                /* Enable form stuff to let the user play with the camera*/
                                //Status->Caption="Found camera";
                                GetPictures->Enabled=true;
                                DeletePictures->Enabled=true;
                                Advanced1->Enabled=true;

                                return true;

    			}
    		}
    	}


        /* Did NOT detect the camera - ensure all camera-related function buttons are disabled */

        StatusBar->SimpleText="Did not detect camera";
        GetPictures->Enabled=false;
        DeletePictures->Enabled=false;
        Advanced1->Enabled=false;
        NumPicturesLabel->Caption="??";

        return false;
}
void __fastcall TForm1::ReQueryClick(TObject *Sender)
{
        OpenCam();
}
//---------------------------------------------------------------------------
void __fastcall TForm1::GetPicturesClick(TObject *Sender)
{
        StatusBar->SimpleText="Preparing to download pictures";
        StatusBar->Repaint();

        GetPictures->Enabled=false;
        DeletePictures->Enabled=false;
        Advanced1->Enabled=false;
        ReQuery->Enabled=false;

        int err=DownloadPictures();
        if (err==0)
        {
                StatusBar->SimpleText="Done.";
        }
        else if (err==1)
        {
                StatusBar->SimpleText="File save cancelled.";
        }
        else
        {
                StatusBar->SimpleText="Error occurred during file download.";
        }

        GetPictures->Enabled=true;
        DeletePictures->Enabled=true;
        Advanced1->Enabled=true;
        ReQuery->Enabled=true;



}
//---------------------------------------------------------------------------
void __fastcall TForm1::Exit1Click(TObject *Sender)
{

        usb_release_interface(cam, interface);
        usb_close(cam);


        Form1->Close();
        exit(0);
}
//---------------------------------------------------------------------------



void __fastcall TForm1::DeletePicturesClick(TObject *Sender)
{
        int err;

        GetPictures->Enabled=false;
        DeletePictures->Enabled=false;
        Advanced1->Enabled=false;
        ReQuery->Enabled=false;

        /* Out, 0x52 (delete all pictures) */
        usb_control_msg(cam, 0x41, 0x52, 0x0000, 0x0000, NULL, 0, 5000);
        err=wait_for_ready();
        if (err)
        {
                StatusBar->SimpleText="Timeout waiting for ready.";
                //return;
        }
        _sleep(1);
        OpenCam();
        ReQuery->Enabled=true;
        /* Note: The camera's behavior after picture deletion is inconsistent. It may be 'Ready'*/
        /* again immediately, or appear to shut off after a few moments. But it is still */
        /* active and accessible! */
}
//---------------------------------------------------------------------------

void __fastcall TForm1::SendReset1Click(TObject *Sender)
{
        cam_reset();
}
//---------------------------------------------------------------------------

void __fastcall TForm1::SendEnable1Click(TObject *Sender)
{
        cam_enable();
}
//---------------------------------------------------------------------------

void __fastcall TForm1::DeviceStatus1Click(TObject *Sender)
{
        char message[1000];
        char fwrev_buf[5];
        char status_buf[7];
        char serial_buf[9];
        char serial_buf0[4];
        char mode_buf[1];
        char mapping_buf[1];
        char ena_buf[4];
        /*In, read firmware revision (0x20) */
        usb_control_msg(cam, 0xc1, 0x20, 0x0000, 0x0000, fwrev_buf, 5, 5000); //0x20

        /*In, read ready status (0x21) */
        //usb_control_msg(cam, 0xc1, 0x21, 0x0000, 0x0000, status_buf, 1, 5000);
        usb_control_msg(cam, 0xc1, 0x21, 0x0000, 0x0003, status_buf, 7, 5000);

        /*In, read device serial number*/
        usb_control_msg(cam, 0xc1, 0x2d, 0x0000, 0x0000, serial_buf0, 3, 5000); /* First three characters, seems to be "BA" followed by a number */
        usb_control_msg(cam, 0xc1, 0x2d, 0x0000, 0x0001, serial_buf, 8, 5000); /* Second part of serial number, as used in cam-enable routine */
        serial_buf0[3] = 0;
        serial_buf[8] = 0; /* Terminate serial # string */

        /*In, read enable status */
        usb_control_msg(cam, 0xc1, 0x2d, 0x0000, 0x0002, ena_buf, 1, 5000); /* proper 'Get enable status' command - could also read address 0x0ee7 directly */
        /*In, read mode status */
        usb_control_msg(cam, 0xc1, 0x00, 0x0000, 0x2000, mode_buf, 1, 5000);
        usb_control_msg(cam, 0xc1, 0x00, 0x0000, 0x2c07, mapping_buf, 1, 5000);

        sprintf (message, "Firmware version: %02x %02x %02x %02x %02x \nSerial #: %s%s\nReady status: %02x\nOperation mode: %02x\nEnable status: %02x\nComposite status: %02x %02x %02x %02x %02x %02x %02x\nMapper (0x2c07): %02x",
                (unsigned int) ((char *)fwrev_buf)[0],
                (unsigned int) ((char *)fwrev_buf)[1],
                (unsigned int) ((char *)fwrev_buf)[2],
                (unsigned int) ((char *)fwrev_buf)[3],
                (unsigned int) ((char *)fwrev_buf)[4],
                serial_buf0,
                serial_buf,
                (unsigned int) ((char *)status_buf)[0],
                (unsigned int) ((char *)mode_buf)[0],
                (unsigned int) ((char *)ena_buf)[0],
                (unsigned int) ((char *)status_buf)[0],
                (unsigned int) ((char *)status_buf)[1],
                (unsigned int) ((char *)status_buf)[2],
                (unsigned int) ((char *)status_buf)[3],
                (unsigned int) ((char *)status_buf)[4],
                (unsigned int) ((char *)status_buf)[5],
                (unsigned int) ((char *)status_buf)[6],
                (unsigned int) ((char *)mapping_buf)[0]
                );

        Application->MessageBox(message,"Device status and information", MB_OK);
}
//---------------------------------------------------------------------------


void TForm1::ReadRange(int start, int end)
{

        FILE *fout;
        int err;
        char status[256];
        char response[128]; /* If you change the size of the block to read here, also remember to change it in the loop and USB read commands! */
                                /* Calls to this function must then end at a multiple of this number! */

        int result=SaveDialog2->Execute();
        Form1->Repaint();

        if (result==0)
        {
                StatusBar->SimpleText="File save cancelled.";
                return;
        }

        fout=fopen(SaveDialog2->FileName.c_str(), "rb");
        if (fout)
        {
                /* File exists! */
                err=Application->MessageBox("File already exists. Overwrite it?", NULL, MB_YESNOCANCEL);
        }
        fclose(fout);

        if (err==2)     /* 'Cancel' button */
        {
                return; /* Return (abort file downloading entirely) */
        }
        else if (err==7) /* 'No' button */
        {
        }
        else
        {
                /* Download file */
                fout=fopen(SaveDialog2->FileName.c_str(), "wb");
                if (!fout)
                {
                        Application->MessageBox("Unable to open the file for writing.", NULL, MB_OK);
                        return;
                }

                /* Make sure we start on the first bank */
                usb_control_msg(cam, 0x41, 0x00, 0x0000, 0x2c07, NULL, 0, 5000);
                for (int i=start; i<end; i=i+128)
                {
                        if (i==0x10000)
                        {
                                /* Switch memory bank and keep reading */
                                usb_control_msg(cam, 0x41, 0x00, 0x0001, 0x2c07, NULL, 0, 5000);
                        }
                        if (i>0xffff)
                        {
                                /* Read from $8000 ~ $ffff (bank switched) for addresses > $ffff */
                                usb_control_msg(cam, 0xc1, 0x00, 0x0000, i-0x8000, response, 128, 5000);
                        }
                        else
                        {
                                /*Read address as given*/
                                usb_control_msg(cam, 0xc1, 0x00, 0x0000, i, response, 128, 5000);
                        }
		        //fprintf(fout, "%c", response);
                        fwrite(response, 128, 1, fout);
                        sprintf(status, "Reading out memory... 0x%04X/0x%04X", i, end);
                        StatusBar->SimpleText=status;
                        StatusBar->Repaint();
                }
                fflush(fout);
                fclose(fout);
                StatusBar->SimpleText="Done.";
        }

}
void __fastcall TForm1::ROM1Click(TObject *Sender)
{
        ReadRange(0x8000,0x17fff);
        /* Note - can only read the low half of the ROM by this method. However, the high half is blank anyway */
}
//---------------------------------------------------------------------------

void __fastcall TForm1::Registers1Click(TObject *Sender)
{
        ReadRange(0x2000,0x2fff);
}
//---------------------------------------------------------------------------



void writecmd(unsigned char byte)
{
        usb_control_msg(cam, 0x41, 0x00, 0x0003, 0x2423, NULL, 0, 5000);
        usb_control_msg(cam, 0x41, 0x00, 0x0002, 0x2423, NULL, 0, 5000);
        usb_control_msg(cam, 0x41, 0x00, 0x000a, 0x2423, NULL, 0, 5000);
        usb_control_msg(cam, 0x41, 0x00, byte,   0x2420, NULL, 0, 5000);
        usb_control_msg(cam, 0x41, 0x00, 0x0002, 0x2423, NULL, 0, 5000);
}


void writeaddr(unsigned char byte)
{
        usb_control_msg(cam, 0x41, 0x00, 0x0006, 0x2423, NULL, 0, 5000);
        usb_control_msg(cam, 0x41, 0x00, byte,   0x2420, NULL, 0, 5000);
        usb_control_msg(cam, 0x41, 0x00, 0x0002, 0x2423, NULL, 0, 5000);
}

char readreg(unsigned int addr)
{
        char buf[1];
        usb_control_msg(cam, 0xc1, 0x00, 0x0000, addr, buf, 1, 5000);
        return buf[0];
}


void __fastcall TForm1::All1Click(TObject *Sender)
{

        FILE *data, *ecc;
        int err=0;
        char status[256];
        char dataname[512];
        char eccname[512];

        int result=SaveDialog2->Execute();
        Form1->Repaint();

        if (result==0)
        {
                StatusBar->SimpleText="File save cancelled.";
                return;
        }
        sprintf(dataname, "%s", SaveDialog2->FileName.c_str());
        sprintf(eccname, "%s.ecc", SaveDialog2->FileName.c_str());
        data=fopen(dataname, "rb");
        ecc=fopen(eccname, "rb");
        if (data || ecc)
        {
                /* File exists! */
                err=Application->MessageBox("File already exists. Overwrite it?", NULL, MB_YESNOCANCEL);
        }
        fclose(data);
        fclose(ecc);

        if (err==2)     /* 'Cancel' button */
        {
                return; /* Return (abort file downloading entirely) */
        }
        else if (err==7) /* 'No' button */
        {
        }
        else
        {
                /* Write files */
                data=fopen(dataname, "wb");
                ecc=fopen(eccname, "wb");

                if (!data || !ecc)
                {
                        Application->MessageBox("Unable to open the file(s) for writing.", NULL, MB_OK);
                        return;
                }

                for (int addr=0; addr<=0xffffff; addr+=512)
                {


                        unsigned char bank = (addr >> 8) & 1;
                        unsigned char a0_7 = addr & 0xff;
                        unsigned char a9_16 = (addr >> 9) & 0xff;
                        unsigned char a17_23 = (addr >> 17) & 0xff;   /* actually go to a24 */
                        unsigned int i;
                        unsigned char byte;

                        /* get ready to talk to the flash */
                        byte = readreg(0x2026);
                        usb_control_msg(cam, 0x41, 0x00, byte, 0x2026, NULL, 0, 5000);
                        byte = readreg(0x2019);
                        usb_control_msg(cam, 0x41, 0x00, byte | 0x02, 0x2019, NULL, 0, 5000);
                        usb_control_msg(cam, 0x41, 0x00, 0x0003, 0x2423, NULL, 0, 5000);
                        usb_control_msg(cam, 0x41, 0x00, 0x0002, 0x2423, NULL, 0, 5000);

                        /* Do a read, <READ0> or <READ1> */
                        writecmd(bank);
                        writeaddr(a0_7);
                        writeaddr(a9_16);
                        writeaddr(a17_23);

                        for(i=0; i<512; i++)
                        {
                                byte = readreg(0x2420);
                                fputc(byte, data);
                        }

                        for(i=0; i<16; i++)
                        {     /* get ecc - 16 bytes */
                                byte = readreg(0x2420);
                                fputc(byte, ecc);
                        }
                        fflush(data);
                        fflush(ecc);

                        /* reset */
                        writecmd(0xFF);

                        /* done talking to the flash for a bit */
                        usb_control_msg(cam, 0x41, 0x00, 0x0001, 0x2423, NULL, 0, 5000);
                        byte = readreg(0x2026);
                        usb_control_msg(cam, 0x41, 0x00, byte | 0x01, 0x2026, NULL, 0, 5000);
                        byte = readreg(0x2019);
                        usb_control_msg(cam, 0x41, 0x00, byte & 0xfd, 0x2019, NULL, 0, 5000);


                        sprintf(status, "Reading out FLASH. This will take all day... 0x%06X/0xFFFFFF", addr);
                        StatusBar->SimpleText=status;
                        StatusBar->Repaint();
                }
                fclose(data);
                fclose(ecc);
                StatusBar->SimpleText="Done.";
        }


}
//---------------------------------------------------------------------------

void __fastcall TForm1::RAM1Click(TObject *Sender)
{
        ReadRange(0x0000,0x1800);
}
//---------------------------------------------------------------------------

void __fastcall TForm1::SendcheatersEnable1Click(TObject *Sender)
{
        cam_force_enable();
}
//---------------------------------------------------------------------------


void __fastcall TForm1::N0Idle1Click(TObject *Sender)
{
        usb_control_msg(cam, 0x41, 0x32, 0x0000, 0x0001, NULL, 0, 5000);
}
//---------------------------------------------------------------------------

void __fastcall TForm1::N1DigitalStillCamera1Click(TObject *Sender)
{
        usb_control_msg(cam, 0x41, 0x32, 0x0000, 0x0002, NULL, 0, 5000);
}
//---------------------------------------------------------------------------

void __fastcall TForm1::N2VideoClipviareq311Click(TObject *Sender)
{
        usb_control_msg(cam, 0x41, 0x31, 0x0000, 0x0003, NULL, 0, 5000);
}
//---------------------------------------------------------------------------

void __fastcall TForm1::N3VideoClipviareq321Click(TObject *Sender)
{
        usb_control_msg(cam, 0x41, 0x32, 0x0000, 0x0003, NULL, 0, 5000);
}
//---------------------------------------------------------------------------

void __fastcall TForm1::N3Webcamviareq311Click(TObject *Sender)
{
        usb_control_msg(cam, 0x41, 0x31, 0x0000, 0x0004, NULL, 0, 5000);
}
//---------------------------------------------------------------------------

void __fastcall TForm1::N3Webcamviareq321Click(TObject *Sender)
{
        usb_control_msg(cam, 0x41, 0x32, 0x0000, 0x0004, NULL, 0, 5000);
}
//---------------------------------------------------------------------------

void __fastcall TForm1::N4Upload1Click(TObject *Sender)
{
        usb_control_msg(cam, 0x41, 0x32, 0x0000, 0x0005, NULL, 0, 5000);
}
//---------------------------------------------------------------------------

void __fastcall TForm1::LibUSBVersion2Click(TObject *Sender)
{
        //struct usb_version *usb_get_version(void);
        usb_version *ver;
        ver=usb_get_version();
        char msg[128];
        sprintf(msg, "DLL version: %d.%d.%d.%d\nDriver version: %d.%d.%d.%d", ver->dll.major, ver->dll.minor, ver->dll.micro, ver->dll.nano, ver->driver.major, ver->driver.minor, ver->driver.micro, ver->driver.nano); //
        Application->MessageBox( msg ,"libusb-win32 version", MB_OK);
}
//---------------------------------------------------------------------------

void __fastcall TForm1::Author1Click(TObject *Sender)
{
        AboutForm->Show();
}
//---------------------------------------------------------------------------

void __fastcall TForm1::MemoryWatcher1Click(TObject *Sender)
{
        MemoryWatcherForm->Show();
}
//---------------------------------------------------------------------------

void poker (unsigned int addr, unsigned int value)
{
        // Writes data to a camera memory location. Use with caution :-)
        usb_control_msg(cam, 0x41, 0x00, value, addr, NULL, 0, 5000);
}


void __fastcall TForm1::CameraUSBInterface1Click(TObject *Sender)
{
        char msg[128];
        sprintf(msg, "Configuration: %X\nInterface: %X\nAlt. setting: %X\nEndpoint address: 0x%X\n", configuration, interface, altsetting, bulk_endpoint); //
        Application->MessageBox( msg ,"USB Bulk endpoint detection", MB_OK);
}
//---------------------------------------------------------------------------

void __fastcall TForm1::TakePictureNow1Click(TObject *Sender)
{
        HowManyForm->Show();
}
//---------------------------------------------------------------------------



void __fastcall TForm1::QuickSnapBtnClick(TObject *Sender)
{
        usb_control_msg(cam, 0x41, 0x51, 0x0000, 0x0000, NULL, 0, 5000);
}
//---------------------------------------------------------------------------

void takesomepics (int num)
{
        if (num<1)
        {
                num=1;
        }
        usb_control_msg(cam, 0x41, 0x51, num-1, 0x0000, NULL, 0, 5000);
}


